# -*- coding: utf-8 -*-

"""
Created on Fri Jan  7 16:15:07 2022

@author: pwiechen
"""

#%%
import numpy as np   
import os
import netCDF4 as nc4
import pandas as pd            
import matplotlib.pyplot as plt 
from datetime import datetime, timedelta
from scipy.ndimage import uniform_filter1d

dir_path = os.path.dirname(os.path.abspath(__file__))
os.chdir(dir_path)

print(dir_path)
#%% OSSI 1

def ossi_data_reader(path):
    
    '''
    Function to read all WLOG_XXX files in a certain subfolder.
    Make sure that only WLOG_XXX files are in this folder and no other files.
    Only WLOG_XXX files with minimally 2 rows are appended to the dataframe.
    A correct WLOG_XXX file should contain a first line with OSSI configuration, and a second line (third row) with starting time
    Timestep and sampling frequency are retrieved from the first row. Starting time from the next row
    Returns a dataframe with a time column and pressure column in Pa
    '''
    
    ossi = pd.DataFrame({
        'time': [],
        'pressure':[]})

    directory = str(path)

    for filename in os.listdir(directory):
        f = os.path.join(directory, filename)
    
        # checking if it is a file
        if os.path.isfile(f):
            print('Currently concatenating file ' + f)
            ossi_raw = pd.read_csv(f, header=None, nrows=4, sep=',')
            if len(ossi_raw.index) > 2:
                t_0 = datetime(int(str(20) + ossi_raw[0][1][1:]),int(ossi_raw[1][1][1:]),int(ossi_raw[2][1][1:]),int(ossi_raw[3][1][1:]),int(ossi_raw[4][1][1:]),int(ossi_raw[5][1][1:]))
                dt = 1/float(ossi_raw[6][0][1:])
                ossi_tot = pd.read_csv(f, skiprows=3, usecols=[0,1,2,3,4,5,6,7,8,9,10,11], header=None, sep=',', skipinitialspace=True).values.flatten()
                ossi_temp = pd.DataFrame({
                    'time':np.array([t_0 + timedelta(seconds=dt*i) for i in range(len(ossi_tot))]),
                    'pressure':ossi_tot})
            
                ossi_temp.dropna(inplace=True)
                ossi_temp['pressure'] = ossi_temp['pressure'] * 10**5 #bar to Pa
                
                ossi = pd.concat([ossi,ossi_temp], ignore_index=True)
                
                
    ossi['pressure'] = pd.to_numeric(ossi['pressure'])
    ossi['time'] = pd.to_datetime(ossi['time'])            
 
    return ossi

#%% Load Rijkswaterstaat Data


eta_HvH_SCHE_RWS = pd.read_csv(r'../../../Supplementary data/dataset_eta_HvH_SCHE_RWS.csv',index_col=0,parse_dates=['time'])  # Water level at Hoek van Holland and Scheveningen, and estimate of water level at field site
eta_HvH_SCHE_RWS['time_unix'] = pd.to_numeric(eta_HvH_SCHE_RWS['time'])/10**9 - 3600 # Conversion to winter time for field site

air_pressure_HvH_Voor_KNMI = pd.read_csv(r'../../../Supplementary data/dataset_air_pressure_HvH_Voorschoten_KNMI.csv',index_col=0,parse_dates=['time']) # Air pressure at Hoek van Holland and Voorschoten
air_pressure_HvH_Voor_KNMI['pressure_sand_engine'] = 0.5*(air_pressure_HvH_Voor_KNMI['pressure_HvH']+air_pressure_HvH_Voor_KNMI['pressure_voorschoten']) # Air pressure estimate at field site
air_pressure_HvH_Voor_KNMI['time_unix'] = pd.to_numeric(air_pressure_HvH_Voor_KNMI['time'])/10**9 - 3600 # Conversion to winter time for field site


#%% Create OSSI dataset

t_b_experiment = datetime(2021,11,6,10,0,0).timestamp()
t_e_experiment = datetime(2021,11,8,13,0,0).timestamp()

s01_ossi_01_raw = ossi_data_reader('raw_data_s01_ossi_01')
s01_ossi_01_raw['time_unix'] = pd.to_numeric(s01_ossi_01_raw['time'].values)*10.0**-9 - 3600

#%% Interpolate to continuous timeseries to fill the data gap at the last 50s at the end of each day that the OSSI needs to write data from the instrument to the memory card

t_interp = np.around(np.arange(s01_ossi_01_raw['time_unix'].values[0], s01_ossi_01_raw['time_unix'].values[-1] + 0.1, step=0.1), 1)

s01_ossi_01 = pd.DataFrame(data={
                    'time_unix': t_interp,
                    'pressure': np.interp(t_interp, s01_ossi_01_raw['time_unix'], s01_ossi_01_raw['pressure'])})

# Replace the datapoints of the last 2 minutes and first 2 minutes of each day with nans
# We choose the first and last 2 minutes to be consistent with the main November-December deployment. 
# NB: It does not really matter, as this 4 min period is during low tide. 

t_netcdf_1 = pd.to_datetime(s01_ossi_01['time_unix'].values+3600, unit='s')

s01_ossi_01['pressure'][np.where((t_netcdf_1.hour == 23) & (t_netcdf_1.minute >= 58))[0]] = np.nan
s01_ossi_01['pressure'][np.where((t_netcdf_1.hour == 0) & (t_netcdf_1.minute <= 1))[0]] = np.nan

# Interpolate air pressure dataset to the timestep of the OSSI 

s01_ossi_01['air_pressure'] = np.interp(s01_ossi_01['time_unix'],xp=air_pressure_HvH_Voor_KNMI['time_unix'], fp=air_pressure_HvH_Voor_KNMI['pressure_HvH'])

#%% Compute the OSSI offset based on air pressure between 2021-11-06 21:00 - 24:00 (Local time)

t_cal_b = datetime(2021,11,6,21,0,0).timestamp()
t_cal_e = datetime(2021,11,7,0,0,0).timestamp()

pressure_offset = 84530.21593670925 # See other script

# Actual pressure
s01_ossi_01['p'] = s01_ossi_01['pressure'] + pressure_offset

# Pressure with air pressure corrected (apc)
s01_ossi_01['p_apc'] = s01_ossi_01['p'] - s01_ossi_01['air_pressure']

#%% Instrument coordinates 

types_dict = {'description': str,'date':str, 'time_raw':str, 'x_val': float, 'y_val': float, 'z_val': float, '2d_error': float, '3d_error': float}

instrument_height_ossi = 2.54*0.145
max_error = 0.05

gps_20211106 = pd.read_csv(r'../../GPS/20211106/20211106_INSTRUMENTS.txt', usecols=[0,1,2,3,4,5,6,7], index_col=False,delim_whitespace=True, skipinitialspace=True, names=['description','date', 'time_raw', 'x_val', 'y_val', 'z_val', '2d_error', '3d_error'], dtype=types_dict, parse_dates={'time':['date', 'time_raw']})
gps_20211106 = gps_20211106.drop(gps_20211106[gps_20211106['3d_error'] > max_error].index)

S01_OSSI_01 = np.mean(gps_20211106[gps_20211106.description.str.contains('S01_OSSI_01_TOP')].values[:,2:], axis=0)
h_bed_S01_OSSI_01 = np.mean(gps_20211106[gps_20211106.description.str.contains('S01_OSSI_01_BED')].values[:,2:], axis=0)
S01_OSSI_01[2] -= instrument_height_ossi

#%% Creation of Netcdf

s01_ossi_netcdf = nc4.Dataset(r'20211106_1000_20211108_1300_ossi.nc','w', format='NETCDF4')

s01_ossi_netcdf.createDimension('time', None)
s01_ossi_netcdf.createDimension('loc', 0)

x_ossi = s01_ossi_netcdf.createVariable('x', 'f8', 'loc')
y_ossi = s01_ossi_netcdf.createVariable('y', 'f8', 'loc')
z_ossi = s01_ossi_netcdf.createVariable('z', 'f8', 'loc')

time = s01_ossi_netcdf.createVariable('time', 'f8', 'time')

ap_var = s01_ossi_netcdf.createVariable('ap', 'f8', 'time')

p_1 = s01_ossi_netcdf.createVariable('p_1', 'f8', 'time')
p_oc_1 = s01_ossi_netcdf.createVariable('p_oc_1', 'f8', 'time')
p_apc_1 = s01_ossi_netcdf.createVariable('p_apc_1', 'f8', 'time')

x_ossi[:] = '%.4f' % round(S01_OSSI_01[0],4)
y_ossi[:] = '%.4f' % round(S01_OSSI_01[1],4)
z_ossi[:] = '%.4f' % round(S01_OSSI_01[2],4)

time[:] = np.around(s01_ossi_01['time_unix'][np.where(s01_ossi_01['time_unix'] >= t_b_experiment)[0][0]:np.where(s01_ossi_01['time_unix'] >= t_e_experiment)[0][0]], 1)

ap_var[:] = s01_ossi_01['air_pressure'][np.where(s01_ossi_01['time_unix'] >= t_b_experiment)[0][0]:np.where(s01_ossi_01['time_unix'] >= t_e_experiment)[0][0]]

p_1[:] = s01_ossi_01['pressure'][np.where(s01_ossi_01['time_unix'] >= t_b_experiment)[0][0]:np.where(s01_ossi_01['time_unix'] >= t_e_experiment)[0][0]]
p_oc_1[:] = s01_ossi_01['p'][np.where(s01_ossi_01['time_unix'] >= t_b_experiment)[0][0]:np.where(s01_ossi_01['time_unix'] >= t_e_experiment)[0][0]]
p_apc_1[:] = s01_ossi_01['p_apc'][np.where(s01_ossi_01['time_unix'] >= t_b_experiment)[0][0]:np.where(s01_ossi_01['time_unix'] >= t_e_experiment)[0][0]]

#Add global attributes

s01_ossi_netcdf.description = 'OSSI pressure sensor data from the November storm deployment, 10:00 November 6 2021 - 13:00 November 8 2022, for the Realdune - Reflex experiments on the Sand Engine, the Netherlands.'
today = datetime.today()
s01_ossi_netcdf.history = "Created on [YYYY-mm-dd]: " + today.strftime("%Y-%m-%d")

#Add local attributes to variable 

x_ossi.units = 'm RDNAP'
y_ossi.units = 'm RDNAP'
z_ossi.units = 'm RDNAP'

time.units = 's'

ap_var.units = 's'

p_1.units = 'Pa'
p_oc_1.units = 'Pa'
p_apc_1.units = 'Pa'

#Add local descriptions

x_ossi.description = 'x-coordinate sensor in RDNAP [m]'
y_ossi.description = 'y-coordinate sensor in RDNAP [m]'
z_ossi.description = 'z-coordinate sensor in RDNAP [m]'

time.description = 'time in unix epoch time (seconds after 1970-01-01 00:00:00 UTC) [s]'

ap_var.description = 'Air pressure at the Hoek van Holland station (measured) [Pa]'

p_1.description = 'Raw pressure at sensor S01_OSSI_01 [Pa]'
p_oc_1.description = 'pressure at sensor S01_OSSI_01, corrected for the offset of the instrument [Pa]'
p_apc_1.description = 'pressure at sensor S01_OSSI_01, corrected for air pressure and offset of the instrument [Pa]'

s01_ossi_netcdf.close()

#%% Figure

november_ossi = nc4.Dataset(r'20211106_1000_20211108_1300_ossi.nc','r')

#%%
w_1_begin = datetime(2021,11,6,10,0,0).timestamp()
w_1_end = datetime(2021,11,6,23,58,0).timestamp()
w_2_begin = datetime(2021,11,7,0,2,0).timestamp()
w_2_end = datetime(2021,11,7,23,58,0).timestamp()
w_3_begin = datetime(2021,11,8,0,2,0).timestamp()
w_3_end = datetime(2021,11,8,13,0,0).timestamp()

fig, (ax1, ax2, ax3) = plt.subplots(nrows=3, sharex=True, figsize=(4.7747*2, 4.7747*6/5))

ax1.plot(pd.to_datetime(s01_ossi_01_raw['time_unix'][:] + 3600, unit='s'),s01_ossi_01_raw['pressure'][:], label='Pressure OSSI raw')
ax1.plot(pd.to_datetime(november_ossi['time'][:] + 3600, unit='s'),november_ossi['p_1'][:], label='Pressure OSSI netcdf', linestyle='dotted')

ax2.plot(pd.to_datetime(november_ossi['time'][:] + 3600, unit='s'),november_ossi['p_oc_1'][:], label='Corrected pressure OSSI')
ax2.plot(pd.to_datetime(november_ossi['time'][:] + 3600, unit='s'),november_ossi['ap'][:], label='Air pressure Sand Engine')

ax2.axvspan(datetime(2021,11,6,21,0,0),datetime(2021,11,7,0,0,0), color='C3', alpha = 0.2, label='Calibration interval')

ax2.set(xlim = (datetime(2021,11,6,9,0,0),datetime(2021,11,8,14,0,0)), ylabel='Pressure [Pa]')

ax2.ticklabel_format(axis='y', style='scientific', scilimits=(4,4))

ax2.legend(loc='upper left')
ax2.grid()

window_size = 30*60*10 #30 minute window for moving average

ax3.plot(pd.to_datetime(november_ossi['time'][np.where(november_ossi['time'][:]>=w_1_begin)[0][0]:np.where(november_ossi['time'][:]>=w_1_end)[0][0]] + 3600, unit='s'), november_ossi['z'][:] + uniform_filter1d(november_ossi['p_apc_1'][np.where(november_ossi['time'][:]>=w_1_begin)[0][0]:np.where(november_ossi['time'][:]>=w_1_end)[0][0]]/(1025*9.81),window_size), color='C0', label=r'$\eta_{S01}$')
ax3.plot(pd.to_datetime(november_ossi['time'][np.where(november_ossi['time'][:]>=w_2_begin)[0][0]:np.where(november_ossi['time'][:]>=w_2_end)[0][0]] + 3600, unit='s'), november_ossi['z'][:] + uniform_filter1d(november_ossi['p_apc_1'][np.where(november_ossi['time'][:]>=w_2_begin)[0][0]:np.where(november_ossi['time'][:]>=w_2_end)[0][0]]/(1025*9.81),window_size), color='C0')
ax3.plot(pd.to_datetime(november_ossi['time'][np.where(november_ossi['time'][:]>=w_3_begin)[0][0]:] + 3600, unit='s'), november_ossi['z'][:] + uniform_filter1d(november_ossi['p_apc_1'][np.where(november_ossi['time'][:]>=w_3_begin)[0][0]:]/(1025*9.81),window_size), color='C0')

ax3.plot(pd.to_datetime(eta_HvH_SCHE_RWS['time_unix'] + 3600, unit='s'), eta_HvH_SCHE_RWS['eta_se_estimate_act'], color='C1', label=r'$\eta_{RWS}$')
ax3.set(xlabel='Date [MM-DD HH]', ylabel='Elevation [m NAP]')
ax3.legend(loc='upper left')
ax3.grid()

plt.tight_layout()


#%%

november_ossi.close()

#%%